/************************************************************************************************************\

Module Name:    LH264VideoCodec.h

Description:    Header declaration file of the video codec Liberatus interface.

References:     LH264VideoCodec.doc

    Copyright (c) 2012, Matrox Graphics Inc.
    All Rights Reserved.

\************************************************************************************************************/

#ifndef INC_LH264VIDEOCODEC_H
#define INC_LH264VIDEOCODEC_H

// -----------------------------------------------------------------------------------------------------------
//                                   I N C L U D E S   A N D   U S I N G S
// -----------------------------------------------------------------------------------------------------------

#include "LPixelFormat.h"
#include "Liberatus.h"

typedef struct LH264TraceOptions_OBJECT*  LH264_TraceOptions;
typedef struct LH264DECODER_OBJECT*  LH264D_Handle;
typedef struct LH264ENCODER_OBJECT*  LH264E_Handle;

// -----------------------------------------------------------------------------------------------------------
//                                   C O N S T A N T S   A N D   T Y P E S
// -----------------------------------------------------------------------------------------------------------

#pragma pack(push, 4)

// Constant and types are defined in the following order:
// 1. Generic (Decoder and Encoder) types
// 2. Decoder types
// 3. Encoder types

// *********************************************
// 1. Generic (Decoder and Encoder) types
// *********************************************
//
/************************************************************************************************************\

Enum:         LH264_Priority

Description:  Stream priorities supported.

Comments:

\************************************************************************************************************/
typedef enum tagLH264_Priority
{
    LH264_Priority_NONREALTIME,
    LH264_Priority_REALTIME,
    LH264_Priority_REALTIMELOWLATENCY
} LH264_Priority;

/************************************************************************************************************\

Enum:         LH264_PictureType

Description:  Picture type.

Comments:

\************************************************************************************************************/
typedef enum tagLH264_PictureType
{
    LH264_PictureType_UNSPECIFIED   = -1,
    // Keep the H.264 types compatible
    LH264_PictureType_P             = 0,
    LH264_PictureType_B             = 1,
    LH264_PictureType_I             = 2,
    // SP = 3,  // Not supported
    // SI = 4,  // Not supported
    LH264_MAX_PICTURE_TYPE          = 5,
    LH264_PictureType_SliceType_Mask= 0xFF,
    // Start the special types from 0x100
    // Start the combined types from 0x10000 to 0x800000
    LH264_PictureType_IDR           = LH264_PictureType_I | 0x10000,
    LH264_PictureType_PSKIP         = LH264_PictureType_P | 0x20000
} LH264_PictureType;

/************************************************************************************************************\

Enum:         LH264_NaluType

Description:  Picture type.

Comments:

\************************************************************************************************************/
typedef enum tagLH264_NaluType
{
    LH264_NaluType_UNSPECIFIED    = 0,
    LH264_NaluType_SLICE          = 1,    //Coded slice of a non-IDR picture slice_layer_without_partitioning_rbsp()
    LH264_NaluType_DPA            = 2,
    LH264_NaluType_DPB            = 3,
    LH264_NaluType_DPC            = 4,
    LH264_NaluType_IDR            = 5,    //Coded slice of an IDR picture slice_layer_without_partitioning_rbsp()
    LH264_NaluType_SEI            = 6,
    LH264_NaluType_SPS            = 7,
    LH264_NaluType_PPS            = 8,
    LH264_NaluType_AUD            = 9,
    LH264_NaluType_EOSEQ          = 10,
    LH264_NaluType_EOSTREAM       = 11,
    LH264_NaluType_FILL           = 12,
    LH264_NaluType_SPSE           = 13,   //Sequence parameter set extension seq_parameter_set_extension_rbsp()
    LH264_NaluType_PREFIXNAL      = 14,   //Prefix NAL unit in scalable extension prefix_nal_unit_rbsp() specified in Annex G
    LH264_NaluType_SSPS           = 15,   //Subset sequence parameter set subset_seq_parameter_set_rbsp() specified in Annex G
    LH264_NaluType_AUXC           = 19,   //Coded slice of an auxiliary coded picture without partitioning slice_layer_without_partitioning_rbsp()
    LH264_NaluType_SEXT           = 20    //Coded slice in scalable extension slice_layer_in_scalable_extension_rbsp() specified in Annex G

    //RESERVED                    16 ... 18  //Reserved
    //RESERVED                    21 ... 23  //Reserved
    //UNSPECIFIED                 24 ... 31  //Unspecified
} LH264_NaluType;

/************************************************************************************************************\

Enum:         LH264_ReferenceMode

Description:  Enumerate the reference modes for B frames.

Comments:

\************************************************************************************************************/
typedef enum tagLH264_ReferenceMode
{
    LH264_ReferenceMode_AUTOMATIC,
    LH264_ReferenceMode_1LEVEL,
    LH264_ReferenceMode_HIERARCHICAL2LEVELNOBREF,
    LH264_ReferenceMode_HIERARCHICAL2LEVEL,
    LH264_ReferenceMode_HIERARCHICAL3LEVEL,
    LH264_ReferenceMode_HIERARCHICAL4LEVEL
} LH264_ReferenceMode;

/************************************************************************************************************\

Enum:         LH264_FieldReferenceMode

Description:  Enumerate the field reference modes.

Comments:

\************************************************************************************************************/
typedef enum tagLH264_FieldReferenceMode
{
    LH264_FieldReferenceMode_SAMEPARITY,
    LH264_FieldReferenceMode_NEARESTFIELD,
    LH264_FieldReferenceMode_ADAPTATIVE
} LH264_FieldReferenceMode;

/************************************************************************************************************\

Enum:         LH264_ChromaSubSampling

Description:  Enumerate the chroma sub-sampling.

Comments:

\************************************************************************************************************/
typedef enum tagLH264_ChromaSubSampling
{
    LH264_ChromaSubSampling_400     = 0,    // Monochrome
    LH264_ChromaSubSampling_420     = 1,    // 4:2:0
    LH264_ChromaSubSampling_422     = 2,    // 4:2:2
    LH264_ChromaSubSampling_444     = 3     // 4:4:4
} LH264_ChromaSubSampling;

/************************************************************************************************************\

Enum:         LH264_HRDMode

Description:  Enumerate the HRD modes.

Comments:

\************************************************************************************************************/
typedef enum tagLH264_HRDMode
{
    LH264_HRDMode_CBR,
    LH264_HRDMode_VBR,
} LH264_HRDMode;

/************************************************************************************************************\

Enum:         LH264_Profile

Description:  Enumerate the profiles supported.

Comments:

\************************************************************************************************************/
typedef enum tagLH264_Profile
{
    LH264_Profile_BASELINE               = 66,   // YUV 4:2:0/8  "Baseline"
    LH264_Profile_MAIN                   = 77,   // YUV 4:2:0/8  "Main"
    LH264_Profile_EXTENDED               = 88,   // YUV 4:2:0/8  "Extended"
    LH264_Profile_HIGH                   = 100,  // YUV 4:2:0/8  "High"
    LH264_Profile_HIGH10                 = 110,  // YUV 4:2:0/10 "High 10"
    LH264_Profile_HIGH422                = 122,  // YUV 4:2:2/10 "High 4:2:2"
    LH264_Profile_HIGH444                = 244,  // YUV 4:4:4/14 "High 4:4:4"
    LH264_Profile_CAVLC444INTRA          = 44    // YUV 4:4:4/14 "CAVLC 4:4:4"
} LH264_Profile;

/************************************************************************************************************\

Enum:         LH264_Level

Description:  Enumerate the levels supported.

Comments:

\************************************************************************************************************/
typedef enum tagLH264_Level
{
    LH264_Level_1B    = 9,
    LH264_Level_1     = 10,
    LH264_Level_1_1   = 11,
    LH264_Level_1_2   = 12,
    LH264_Level_1_3   = 13,
    LH264_Level_2     = 20,
    LH264_Level_2_1   = 21,
    LH264_Level_2_2   = 22,
    LH264_Level_3     = 30,
    LH264_Level_3_1   = 31,
    LH264_Level_3_2   = 32,
    LH264_Level_4     = 40,
    LH264_Level_4_1   = 41,
    LH264_Level_4_2   = 42,
    LH264_Level_5     = 50,
    LH264_Level_5_1   = 51,
    LH264_Level_5_2   = 52
} LH264_Level;

/************************************************************************************************************\

Structure:    LH264_Ratio

Description:  Specify ratio.

Comments:

\************************************************************************************************************/
typedef struct tagLH264_Ratio
{
    MUINT32 uiNumerator;
    MUINT32 uiDenominator;
} LH264_Ratio;

/************************************************************************************************************\

Enum:         LH264_ScanMode

Description:  Specify progressive or interlaced scan mode.

Comments:

\************************************************************************************************************/
typedef enum tagLH264_ScanMode
{
    LH264_ScanMode_UNSPECIFIED,
    LH264_ScanMode_PROGRESSIVE = LH264_ScanMode_UNSPECIFIED,
    LH264_ScanMode_INTERLACED
} LH264_ScanMode;

/************************************************************************************************************\

Enum:         LH264_Field

Description:  Specify the field polarity.

Comments:

\************************************************************************************************************/
typedef enum tagLH264_Field
{
    LH264_FIELDUNSPECIFIED  = 0,
    LH264_TOPFIELD          = 1,
    LH264_BOTTOMFIELD       = 2,
    LH264_FRAME             = (LH264_TOPFIELD | LH264_BOTTOMFIELD)
} LH264_Field;

/************************************************************************************************************\

Enum:         LH264_AspectRatio

Description:  Enumeration representing the sample aspect ratio (pixel aspect ratio).

Comments:     It comes from the H.264 specification.

\************************************************************************************************************/
typedef enum tagLH264_AspectRatio
{
    LH264_AspectRatio_UNSPECIFIED   = 0,
    LH264_AspectRatio_1_1           = 1,
    LH264_AspectRatio_12_11         = 2,
    LH264_AspectRatio_10_11         = 3,
    LH264_AspectRatio_16_11         = 4,
    LH264_AspectRatio_40_33         = 5,
    LH264_AspectRatio_24_11         = 6,
    LH264_AspectRatio_20_11         = 7,
    LH264_AspectRatio_32_11         = 8,
    LH264_AspectRatio_80_33         = 9,
    LH264_AspectRatio_18_11         = 10,
    LH264_AspectRatio_15_11         = 11,
    LH264_AspectRatio_64_33         = 12,
    LH264_AspectRatio_160_99        = 13,
    LH264_AspectRatio_4_3           = 14,
    LH264_AspectRatio_3_2           = 15,
    LH264_AspectRatio_2_1           = 16,
    LH264_AspectRatio_EXTENDEDSAR   = 255
} LH264_AspectRatio;

/************************************************************************************************************\

Enum:         LH264_VideoFormat

Description:  Enumeration of the video formats.

Comments:

\************************************************************************************************************/
typedef enum tagLH264_VideoFormat
{
    LH264_VideoFormat_COMPONENT   = 0,
    LH264_VideoFormat_PAL         = 1,
    LH264_VideoFormat_NTSC        = 2,
    LH264_VideoFormat_SECAM       = 3,
    LH264_VideoFormat_MAC         = 4,
    LH264_VideoFormat_UNSPECIFIED = 5
} LH264_VideoFormat;

/************************************************************************************************************\

Enum:         LH264_ColourPrimaries

Description:  Enumeration of primary color chromaticity coordinate sets.

Comments:

\************************************************************************************************************/
typedef enum tagLH264_ColourPrimaries
{
    LH264_ColourPrimaries_MODE1   = 1,
    LH264_ColourPrimaries_MODE2   = 2,
    LH264_ColourPrimaries_MODE4   = 4,
    LH264_ColourPrimaries_MODE5   = 5,
    LH264_ColourPrimaries_MODE6   = 6,
    LH264_ColourPrimaries_MODE7   = 7,
    LH264_ColourPrimaries_MODE8   = 8,
    LH264_ColourPrimaries_MODE9   = 9,
    LH264_ColourPrimaries_MODE10  = 10,
    LH264_ColourPrimaries_MODE11  = 11,
    LH264_ColourPrimaries_MODE12  = 12,
    LH264_ColourPrimaries_MODE22  = 22
} LH264_ColourPrimaries;

/************************************************************************************************************\

Enum:         LH264_TransferCharacteristics

Description:  Enumeration of optoelectronic transfer functions.

Comments:

\************************************************************************************************************/
typedef enum tagLH264_TransferCharacteristics
{
    LH264_TransferCharacteristics_MODE1   =  1,
    LH264_TransferCharacteristics_MODE2   =  2,
    LH264_TransferCharacteristics_MODE4   =  4,
    LH264_TransferCharacteristics_MODE5   =  5,
    LH264_TransferCharacteristics_MODE6   =  6,
    LH264_TransferCharacteristics_MODE7   =  7,
    LH264_TransferCharacteristics_MODE8   =  8,
    LH264_TransferCharacteristics_MODE9   =  9,
    LH264_TransferCharacteristics_MODE10  = 10,
    LH264_TransferCharacteristics_MODE11  = 11,
    LH264_TransferCharacteristics_MODE12  = 12,
    LH264_TransferCharacteristics_MODE13  = 13,
    LH264_TransferCharacteristics_MODE14  = 14,
    LH264_TransferCharacteristics_MODE15  = 15,
    LH264_TransferCharacteristics_MODE16  = 16,
    LH264_TransferCharacteristics_MODE17  = 17,
    LH264_TransferCharacteristics_MODE18  = 18
} LH264_TransferCharacteristics;

/************************************************************************************************************\

Enum:         LH264_MatrixCoefficients

Description:  Enumeration of matrix coefficients for luma and chroma.

Comments:

\************************************************************************************************************/
typedef enum tagLH264_MatrixCoefficients
{
    LH264_MatrixCoefficients_MODE0   = 0,
    LH264_MatrixCoefficients_MODE1   = 1,
    LH264_MatrixCoefficients_MODE2   = 2,
    LH264_MatrixCoefficients_MODE4   = 4,
    LH264_MatrixCoefficients_MODE5   = 5,
    LH264_MatrixCoefficients_MODE6   = 6,
    LH264_MatrixCoefficients_MODE7   = 7,
    LH264_MatrixCoefficients_MODE8   = 8,
    LH264_MatrixCoefficients_MODE9   = 9,
    LH264_MatrixCoefficients_MODE10  = 10,
    LH264_MatrixCoefficients_MODE11  = 11,
    LH264_MatrixCoefficients_MODE12  = 12,
    LH264_MatrixCoefficients_MODE13  = 13,
    LH264_MatrixCoefficients_MODE14  = 14
} LH264_MatrixCoefficients;

/************************************************************************************************************\

Enum:         LH264_ChromaLocation

Description:  Enumeration of the chroma locations

Comments:

\************************************************************************************************************/
typedef enum tagLH264_ChromaLocation
{
    LH264_ChromaLocation_MODE0   = 0,
    LH264_ChromaLocation_MODE1   = 1,
    LH264_ChromaLocation_MODE2   = 2,
    LH264_ChromaLocation_MODE3   = 3,
    LH264_ChromaLocation_MODE4   = 4,
    LH264_ChromaLocation_MODE5   = 5
} LH264_ChromaLocation;

/************************************************************************************************************\

Enum:         LH264_PictureStruct

Description:  Enumeration of the picture structure information.

Comments:

\************************************************************************************************************/
typedef enum tagLH264_PictureStruct
{
    LH264_PictureStruct_PROGRESSIVEFRAME     = 0,
    LH264_PictureStruct_TOPFIELD             = 1,
    LH264_PictureStruct_BOTTOMFIELD          = 2,
    LH264_PictureStruct_TOPBOTTOMORDER       = 3,
    LH264_PictureStruct_BOTTOMTOPORDER       = 4,
    LH264_PictureStruct_TOPBOTTOMTOPORDER    = 5,
    LH264_PictureStruct_BOTTOMTOPBOTTOMORDER = 6,
    LH264_PictureStruct_FRAMEDOUBLING        = 7,
    LH264_PictureStruct_FRAMETRIPPLING       = 8
} LH264_PictureStruct;

/************************************************************************************************************\

Enum:         LH264_GlobalInfoTypeHeader

Description:  Enumeration of the different types of the global information structure.

Comments:

\************************************************************************************************************/
typedef enum tagLH264_GlobalInfoTypeHeader
{
    LH264_GlobalInfoTypeHeader_STANDARD
} LH264_GlobalInfoTypeHeader;

/************************************************************************************************************\

Structure:    LH264_GlobalInfo

Description:  Global statistics and information in the VC architecture.

Comments:

\************************************************************************************************************/
typedef struct tagLH264_GlobalInfo
{
    MINT32  eType;              // force 32 bits cast of LH264_GlobalInfoTypeHeader
    MUINT32 uiActiveStreamCnt;
    MFLOAT32 fEEUsage[8];
    MFLOAT32 fTEDFUsage[2];
    MFLOAT32 fPEUsage[2];
    MFLOAT32 fProcUsage[8];
    MUINT32 uiWaitReleaseCnt;
    MUINT32 uiRedoCnt;

    MUINT32 auiStreamCnt[128];
} LH264_GlobalInfo;

/************************************************************************************************************\

Enum:         LH264_StreamInfoTypeHeader

Description:  Enumeration of the different structure types containing information on the stream.

Comments:

\************************************************************************************************************/
typedef enum tagLH264_StreamInfoTypeHeader
{
    LH264_StreamInfoTypeHeader_DECODERGENDATA,
    LH264_StreamInfoTypeHeader_ENCODERSEQUENCEDATA,
    LH264_StreamInfoTypeHeader_ENCODERSUBSEQUENCEDATA,
    LH264_StreamInfoTypeHeader_ENCODERNBSLICES,
    LH264_StreamInfoTypeHeader_VUI,
    LH264_StreamInfoTypeHeader_NALHRD,
    LH264_StreamInfoTypeHeader_NALBUFFERINGPERIOD,
    LH264_StreamInfoTypeHeader_VCLHRD,
    LH264_StreamInfoTypeHeader_VCLBUFFERINGPERIOD,
    LH264_StreamInfoTypeHeader_SPSSCALINGLIST,
    LH264_StreamInfoTypeHeader_PPSSCALINGLIST,
    LH264_StreamInfoTypeHeader_ADAPTATIVESCALINGLIST,
    LH264_StreamInfoTypeHeader_DTR,
    LH264_StreamInfoTypeHeader_PRIVATEDATA
} LH264_StreamInfoTypeHeader;

/************************************************************************************************************\

Structure:    LH264_StreamInfoDecoderGenData

Description:  General information that applies to the video stream decoding.

Comments:

\************************************************************************************************************/
typedef struct tagLH264_StreamInfoDecoderGenData
{
    MINT32  eType;                  // force 32 bits cast of LH264_StreamInfoTypeHeader
    MINT32  eProfile;               // force 32 bits cast of LH264_Profile
    MINT32  eLevel;                 // force 32 bits cast of LH264_Level
    MUINT32 uiPictureWidth;
    MUINT32 uiPictureHeight;
    MUINT16 uiPixelDepthChromaPrecision;
    MUINT16 uiPixelDepthLumaPrecision;
    MINT32  ePixelFormat;           // force 32 bits cast of LPixelFormat
    MINT32  eScanMode;              // force 32 bits cast of LH264_ScanMode
    MFLAG32 flDecoderGenDataFlags;
    LRECT32 oPictureCroppedRect;
} LH264_StreamInfoDecoderGenData;

/************************************************************************************************************\

Enum:         LH264_StreamInfoDecoderGenDataFlag

Description:  Enumeration of the different flags available for the decoder generic data.

Comments:

\************************************************************************************************************/
typedef enum tagLH264_StreamInfoDecoderGenDataFlag
{
    LH264_StreamInfoDecoderGenDataFlag_PICTURECROPPEDRECTANGLEPRESENT = 0x00000001
} LH264_StreamInfoDecoderGenDataFlag;

/************************************************************************************************************\

Enum:         LH264_StreamInfoEncoderSequenceDataFlag

Description:  Enumeration of the different flags available for the encoder sequence data.

Comments:

\************************************************************************************************************/
typedef enum tagLH264_StreamInfoEncoderSequenceDataFlag
{
    LH264_StreamInfoEncoderSequenceDataFlag_PICTUREINFOPRESENT              = 0x00000001,
    LH264_StreamInfoEncoderSequenceDataFlag_RATECONTROLPRESENT              = 0x00000002,
    LH264_StreamInfoEncoderSequenceDataFlag_PICTURECROPPEDRECTANGLEPRESENT  = 0x00000004,
    LH264_StreamInfoEncoderSequenceDataFlag_CONTINUOUSINTRAREFRESH          = 0x00000008,
    LH264_StreamInfoEncoderSequenceDataFlag_FILLCROPPEDBOUNDARY             = 0x00000010
} LH264_StreamInfoEncoderSequenceDataFlag;

/************************************************************************************************************\

Enum:         LH264_StreamInfoRateControlFlag

Description:  Enumeration of the different flags available for the rate control.

Comments:

\************************************************************************************************************/
typedef enum tagLH264_StreamInfoRateControlFlag
{
    LH264_StreamInfoRateControlFlag_ENABLEHRD                       = 0x00000001,
    LH264_StreamInfoRateControlFlag_ENABLEFRAMERATECONTROLLER       = 0x00000002,
    LH264_StreamInfoRateControlFlag_FIXEDCODEDPICTURESIZE           = 0x00000004,
    LH264_StreamInfoRateControlFlag_IGNORENONVCL                    = 0x00000008,
    LH264_StreamInfoRateControlFlag_DUPLICATEHRDPARAMETERS          = 0x00000010,
    LH264_StreamInfoRateControlFlag_ENABLEMINCR                     = 0x00000020,
    LH264_StreamInfoRateControlFlag_ENABLESLICEMINCR                = 0x00000040,
} LH264_StreamInfoRateControlFlag;

/************************************************************************************************************\

Structure:    LH264_StreamInfoEncoderSequenceData

Description:  Sequence information that applies to the video stream encoding.

Comments:

\************************************************************************************************************/
typedef struct tagLH264_StreamInfoEncoderSequenceData
{
    MINT32      eType;                    // force 32 bits cast of LH264_StreamInfoTypeHeader
    MFLAG32     flSequenceDataFlags;
    // Picture info (see LH264_StreamInfoEncoderSequenceDataFlag_PICTUREINFOPRESENT)
    MUINT32     uiPictureWidth;
    MUINT32     uiPictureHeight;
    MINT32      ePixelFormat;             // force 32 bits cast of LPixelFormat
    MINT32      eScanMode;                // force 32 bits cast of LH264_ScanMode
    MINT32      eFirstField;              // force 32 bits cast of LH264_Field
    LH264_Ratio oFrameRate;
    MUINT32     eDFDisableIdc;            // Deblocking filter
    MINT32      iDFAlphaC0Param;          // Deblocking filter parameter
    MINT32      iDFBetaParam;             // Deblocking filter parameter
    MFLOAT32    fFAQStrength;             // Frequency Adaptive Quantization strenght
    // Rate control (LH264_StreamInfoEncoderSequenceDataFlag_RATECONTROLPRESENT)
    MFLAG32     flRateControlFlags;
    MUINT32     uiTargetBitRate;
    MFLOAT32    fCorrectionTensor;
    MFLOAT32    fQPMin;
    MFLOAT32    fQPMax;
    MINT32      eHRDMode;                 // force 32 bits cast of LH264_HRDMode
    MUINT32     uiCPBSizeInBits;          // Only in low latency
    MUINT32     uiCPBMaxBitRate;          // Only in VBR HRD mode
    MUINT32     uiBigPictureMaxSize;      // [1..10]
    MUINT32     uiOverrideMinCR;

    // Picture cropped rectangle (see LH264_StreamInfoEncoderSequenceDataFlag_PICTURECROPPEDRECTANGLEPRESENT)
    LRECT32     oPictureCroppedRect;

} LH264_StreamInfoEncoderSequenceData;

/************************************************************************************************************\

Enum:         LH264_StreamInfoEncoderSubSequenceDataFlag

Description:  Enumeration of the different flags available for the encoder sub-sequence data.

Comments:

\************************************************************************************************************/
typedef enum tagLH264_StreamInfoEncoderSubSequenceDataFlag
{
    LH264_StreamInfoEncoderSubSequenceDataFlag_QPPARAMETERPRESENT              = 0x00000001,
    LH264_StreamInfoEncoderSubSequenceDataFlag_REFERENCEPRESENT                = 0x00000002,
    LH264_StreamInfoEncoderSubSequenceDataFlag_CLOSEDP                         = 0x00000010,
    LH264_StreamInfoEncoderSubSequenceDataFlag_ENCODESECONDINTRAFIELDASP       = 0x00000020,
    LH264_StreamInfoEncoderSubSequenceDataFlag_ENABLESMARTSUBSEQUENCE          = 0x00000040,
    LH264_StreamInfoEncoderSubSequenceDataFlag_DELAYFIRSTIDR                   = 0x00000080,
} LH264_StreamInfoEncoderSubSequenceDataFlag;

/************************************************************************************************************\

Structure:    LH264_StreamInfoEncoderSubSequenceData

Description:  Sub-sequence information that applies to the video stream encoding.

Comments:

\************************************************************************************************************/
typedef struct tagLH264_StreamInfoEncoderSubSequenceData
{
    MINT32      eType;                  // force 32 bits cast of LH264_StreamInfoTypeHeader
    MFLAG32     flSubSequenceDataFlags;
    // Qp parameters (LH264_StreamInfoEncoderSubSequenceDataFlag_QPPARAMETERPRESENT)
    MINT32      iQPChroma1Offset;
    MINT32      iQPChroma2Offset;
    // Reference (LH264_StreamInfoEncoderSubSequenceDataFlag_REFERENCEPRESENT)
    MUINT32     uiNbRefForP;
    MINT32      eRefMode;               // force 32 bits cast of LH264_ReferenceMode
    MINT32      eFieldMode;             // force 32 bits cast of LH264_FieldReferenceMode
    MUINT32     uiIdrPeriod;
    MUINT32     uiIPeriod;
    MUINT32     uiPPeriod;
} LH264_StreamInfoEncoderSubSequenceData;

/************************************************************************************************************\

Structure:    LH264_StreamInfoEncoderNbSlices

Description:  Nb Slices information that applies to the video stream.

Comments:

\************************************************************************************************************/
typedef struct tagLH264_StreamInfoEncoderNbSlices
{
    MINT32      eType;                  // force 32 bits cast of LH264_StreamInfoTypeHeader
    MBOOL32     bForceRectangleSlices;
    MUINT32     uiNbSlices;
} LH264_StreamInfoEncoderNbSlices;

/************************************************************************************************************\

Enum:         LH264_StreamInfoVUIFlag

Description:  Enumeration of the different flags available for the VUI.

Comments:

\************************************************************************************************************/
typedef enum tagLH264_StreamInfoVUIFlag
{
    LH264_StreamInfoVUIFlag_ASPECTRATIOPRESENT          = 0x00000001,
    LH264_StreamInfoVUIFlag_OVERSCANPRESENT             = 0x00000002,
    LH264_StreamInfoVUIFlag_OVERSCANAPPROPRIATE         = 0x00000004,
    LH264_StreamInfoVUIFlag_VIDEOSIGNALPRESENT          = 0x00000008,
    LH264_StreamInfoVUIFlag_COLOURDESCRIPTIONPRESENT    = 0x00000010,
    LH264_StreamInfoVUIFlag_CHROMALOCATIONPRESENT       = 0x00000020,
    LH264_StreamInfoVUIFlag_TIMINGINFOPRESENT           = 0x00000040,
    LH264_StreamInfoVUIFlag_FIXEDFRAMERATE              = 0x00000080,
    LH264_StreamInfoVUIFlag_PICTURESTRUCPRESENT         = 0x00000100,
    LH264_StreamInfoVUIFlag_BITSTREAMRESTRICTIONPRESENT = 0x00000200
} LH264_StreamInfoVUIFlag;

/************************************************************************************************************\

Structure:    LH264_StreamInfoVUI

Description:  Video usability information.

Comments:

\************************************************************************************************************/
typedef struct tagLH264_StreamInfoVUI
{
    MINT32  eType;                      // force 32 bits cast of LH264_StreamInfoTypeHeader
    MFLAG32 flVUIFlags;
    MINT32  eAspectRatio;               // force 32 bits cast of LH264_AspectRatio
    MUINT32 uiSARWidth;
    MUINT32 uiSARHeight;
    MINT32  eVideoFormat;               // force 32 bits cast of LH264_VideoFormat
    MBOOL32 bVideoFullRange;
    MINT32  eColourPrimaries;           // force 32 bits cast of LH264_ColourPrimaries
    MINT32  eTransferCharacteristics;   // force 32 bits cast of LH264_TransferCharacteristics
    MINT32  eMatrixCoefficients;        // force 32 bits cast of LH264_MatrixCoefficients
    MINT32  eChromaLocationTopField;    // force 32 bits cast of LH264_ChromaLocation
    MINT32  eChromaLocationBottomField; // force 32 bits cast of LH264_ChromaLocation
    MUINT32 uiNumUnitsInTick;
    MUINT32 uiTimeScale;
    MBOOL32 bMotionVectorsOverPicBoundariesFlag;
    MUINT32 uiMaxBytesPerPicDenom;
    MUINT32 uiMaxBitsPerMBDenom;
    MUINT32 uiLog2MaxMVLengthHorizontal;
    MUINT32 uiLog2MaxMVLengthVertical;
    MUINT32 uiNumReorderFrames;
    MUINT32 uiMaxDecFrameBuffering;
} LH264_StreamInfoVUI;

/************************************************************************************************************\

Enum:         LH264_StreamInfoHRDFlag

Description:  Enumeration of the different flags available for the HRD.

Comments:

\************************************************************************************************************/
typedef enum tagLH264_StreamInfoHRDFlag
{
    LH264_StreamInfoHRDFlag_LOWDELAYHRD = 0x00000001
} LH264_StreamInfoHRDFlag;

/************************************************************************************************************\

Structure:    LH264_StreamInfoHRD

Description:  Hypothetical Reference Decoder information.

Comments:     This information is part of the VUI information, but it is only available with the decoder.

\************************************************************************************************************/
typedef struct tagLH264_StreamInfoHRD
{
    MINT32  eType;          // force 32 bits cast of LH264_StreamInfoTypeHeader
    MFLAG32 flHRDFlag;
    MUINT32 uiCPBCnt;
    MUINT32 auiBitRate[32];
    MUINT32 auiCPBSize[32];
    MBOOL8  abCBR[32];
} LH264_StreamInfoHRD;

/************************************************************************************************************\

Structure:    LH264_StreamInfoBufferingPeriod

Description:  Buffering period information.

Comments:     

\************************************************************************************************************/
typedef struct tagLH264_StreamInfoBufferingPeriod
{
    MINT32  eType;          // force 32 bits cast of LH264_StreamInfoTypeHeader
    MUINT32 uiCPBCnt;
    MUINT32 auiInitialCPBRemovalDelay[32];
} LH264_StreamInfoBufferingPeriod;

/************************************************************************************************************\

Enum:         LH264_StreamInfoScalingListPresentFlag

Description:  Flags specifying the present scaling lists.

Comments:

\************************************************************************************************************/
typedef enum tagLH264_StreamInfoScalingListPresentFlag
{
    LH264_StreamInfoScalingListPresentFlag_YINTRA4X4  = 0x00000001,
    LH264_StreamInfoScalingListPresentFlag_YINTER4X4  = 0x00000002,
    LH264_StreamInfoScalingListPresentFlag_CBINTRA4X4 = 0x00000004,
    LH264_StreamInfoScalingListPresentFlag_CBINTER4X4 = 0x00000008,
    LH264_StreamInfoScalingListPresentFlag_CRINTRA4X4 = 0x00000010,
    LH264_StreamInfoScalingListPresentFlag_CRINTER4X4 = 0x00000020,
    LH264_StreamInfoScalingListPresentFlag_YINTRA8X8  = 0x00000040,
    LH264_StreamInfoScalingListPresentFlag_YINTER8X8  = 0x00000080,
    LH264_StreamInfoScalingListPresentFlag_CBINTRA8X8 = 0x00000100,
    LH264_StreamInfoScalingListPresentFlag_CBINTER8X8 = 0x00000200,
    LH264_StreamInfoScalingListPresentFlag_CRINTRA8X8 = 0x00000400,
    LH264_StreamInfoScalingListPresentFlag_CRINTER8X8 = 0x00000800
} LH264_StreamInfoScalingListPresentFlag;

/************************************************************************************************************\

Structure:    LH264_StreamInfoScalingList

Description:  Custom scaling list.

Comments:

\************************************************************************************************************/
typedef struct tagLH264_StreamInfoScalingList
{
    MINT32  eType;                  // force 32 bits cast of LH264_StreamInfoTypeHeader
    MBOOL32 bUseScalingList;
    MFLAG32 flScalingListPresent;
    MUINT8  auiYIntra4x4[16];
    MUINT8  auiYInter4x4[16];
    MUINT8  auiCbIntra4x4[16];
    MUINT8  auiCbInter4x4[16];
    MUINT8  auiCrIntra4x4[16];
    MUINT8  auiCrInter4x4[16];
    MUINT8  auiYIntra8x8[64];
    MUINT8  auiYInter8x8[64];
    MUINT8  auiCbIntra8x8[64];
    MUINT8  auiCbInter8x8[64];
    MUINT8  auiCrIntra8x8[64];
    MUINT8  auiCrInter8x8[64];
} LH264_StreamInfoScalingList;

/************************************************************************************************************\

Structure:    LH264_StreamInfoAdaptativeScalingList

Description:  Adaptative scaling list.

Comments:

\************************************************************************************************************/
typedef struct tagLH264_StreamInfoAdaptativeScalingList
{
    MINT32  eType;                  // force 32 bits cast of LH264_StreamInfoTypeHeader
} LH264_StreamInfoAdaptativeScalingList;

/************************************************************************************************************\

Structure:    LH264_StreamInfoDTR

Description:  Stream information concerning data to recover.

Comments:

\************************************************************************************************************/
typedef struct tagLH264_StreamInfoDTR
{
    MINT32  eType;                  // force 32 bits cast of LH264_StreamInfoTypeHeader
    MFLAG32 flSlicesToRecover;
    MUINT64 uiFrameId;
} LH264_StreamInfoDTR;


// *********************************************
// 2. Decoder types
// *********************************************
#define     LH264D_MAX_OUTPUT_BUFFER            15

/************************************************************************************************************\

Enum:         LH264D_DecoderMode

Description:  Enumeration of the different decoder modes.

Comments:

\************************************************************************************************************/
typedef enum tagLH264D_DecoderMode
{
    LH264D_DecoderMode_NORMAL,
    LH264D_DecoderMode_LOWLATENCY,
    LH264D_DecoderMode_DESKTOP,
} LH264D_DecoderMode;

/************************************************************************************************************\

Enum:         LH264D_CreateOptTypeHeader

Description:  Enumeration of the different create options types.

Comments:

\************************************************************************************************************/
typedef enum tagLH264D_CreateOptTypeHeader
{
    LH264D_CreateOptTypeHeader_STANDARD
} LH264D_CreateOptTypeHeader;

/************************************************************************************************************\

Enum:         LH264D_CreateOptFlag

Description:  Enumeration of the different create options flags.

Comments:

\************************************************************************************************************/
typedef enum tagLH264D_CreateOptFlag
{
    LH264D_CreateOptFlag_SUPPORTASOFMO               = 0x00000001,
    LH264D_CreateOptFlag_DYNAMICPICTURERESOLUTION    = 0x00000002,
    LH264D_CreateOptFlag_BUFFERAUTOSYNC              = 0x00000004,
    LH264D_CreateOptFlag_PICTUREINFOPRESENT          = 0x00000008,
    LH264D_CreateOptFlag_OPTIMIZEDSLICEDISTRIBUTION  = 0x00000010,
    LH264D_CreateOptFlag_LOWLATENCYLAZYDETECTION     = 0x00000020,
} LH264D_CreateOptFlag;

/************************************************************************************************************\

Structure:    LH264D_CreateOpt

Description:  Options for decode context creation.

Comments:

\************************************************************************************************************/
typedef struct tagLH264D_CreateOpt
{
    MINT32      eType;                  // force 32 bits cast of LH264D_CreateOptTypeHeader
    MUINT64     hDTInput;               // force 64  bits cast of LDeviceThread_Handle
    MUINT64     hDTOutput;              // force 64  bits cast of LDeviceThread_Handle
    MFLAG32     flCreateOptFlags;
    MINT32      eDecoderMode;           // force 32 bits cast of LH264D_DecoderMode
    MINT32      ePriority;              // force 32 bits cast of LH264_Priority
    MINT32      eMaxLevel;              // force 32 bits cast of LH264_Level
    MINT32      eMaxProfile;            // force 32 bits cast of LH264_Profile
    LH264_Ratio oDefaultFrameRate;
    LH264_Ratio oMaxFrameRate;
    MUINT32     uiMaxBitRate;
    MUINT32     uiAutoSyncTimeOut;      // Used only with option LH264D_CreateOptFlag_BUFFERAUTOSYNC
    MUINT32     uiOutputBufferSizeInFrames;
    MUINT32     uiSEIRbspBufferingSize;
    // Picture info (see LH264D_CreateOptFlag_PICTUREINFOPRESENT)
    MUINT32     uiPictureWidth;
    MUINT32     uiPictureHeight;
    MINT32      ePixelFormat;
    MINT32      eScanMode;
    MBOOL       bEntropyCABACMode;
} LH264D_CreateOpt;

/************************************************************************************************************\

Enum:         LH264D_DecodeOptTypeHeader

Description:  Enumeration of the different decode options types.

Comments:

\************************************************************************************************************/
typedef enum tagLH264D_DecodeOptTypeHeader
{
    LH264D_DecodeOptTypeHeader_STANDARD
} LH264D_DecodeOptTypeHeader;

/************************************************************************************************************\

Enum:         LH264D_DecodeOptFlag

Description:  Enumeration of the different decode options flags.

Comments:

\************************************************************************************************************/
typedef enum tagLH264D_DecodeOptFlag
{
    LH264D_DecodeOptFlag_DECODEIPPICTUREONLY        = 0x00000001,
    LH264D_DecodeOptFlag_DECODEREFERENCEPICTUREONLY = 0x00000002,
    LH264D_DecodeOptFlag_NOTIFYENDOFNALU            = 0x00000004,
    LH264D_DecodeOptFlag_NOTIFYENDOFAU              = 0x00000008,
    LH264D_DecodeOptFlag_NOTIFYNONALUSCANREQUIRED   = 0x00000010,
    LH264D_DecodeOptFlag_SEQUENCENUMBERPRESENT      = 0x00000020,
} LH264D_DecodeOptFlag;

/************************************************************************************************************\

Structure:    LH264D_DecodeOpt

Description:  Decode options.

Comments:

\************************************************************************************************************/
typedef struct tagLH264D_DecodeOpt
{
    MINT32  eType;              // force 32 bits cast of LH264D_DecodeOptTypeHeader
    MUINT64 hBuffer;            // force 64 bits cast of LBuffer_Handle
    MUINT64 uiAccessUnitId;
    MUINT32 uiStartOffset;
    MUINT32 uiSize;
    MFLAG32 flDecodeFlags;
    MUINT64 uiSequenceNumber;
} LH264D_DecodeOpt;

/************************************************************************************************************\

Enum:         LH264D_PictureInfoTypeHeader

Description:  Enumeration of the different picture information types.

Comments:

\************************************************************************************************************/
typedef enum tagLH264D_PictureInfoTypeHeader
{
    LH264D_PictureInfoTypeHeader_STANDARD
} LH264D_PictureInfoTypeHeader;

/************************************************************************************************************\

Enum:         LH264D_PictureStatusFlag

Description:  Enumeration of the picture status.

Comments:

\************************************************************************************************************/
typedef enum tagLH264D_PictureStatusFlag
{
    LH264D_PictureStatusFlag_FATALFIRMWAREERROR            = 0x00000001,
    LH264D_PictureStatusFlag_FATALSTREAMERROR              = 0x00000002,
    LH264D_PictureStatusFlag_MISSINGDATA                   = 0x00000004,
    LH264D_PictureStatusFlag_CORRUPTEDDATA                 = 0x00000008,
    LH264D_PictureStatusFlag_UNSUPPORTEDINPUT              = 0x00000010,
    LH264D_PictureStatusFlag_PICTURESSKIPPED               = 0x00000020,
    LH264D_PictureStatusFlag_MISSINGSEIRBSP                = 0x00000040,
    LH264D_PictureStatusFlag_AUFENCE                       = 0x00000080,
} LH264D_PictureStatusFlag;

/************************************************************************************************************\

Enum:         LH264D_PictureInfoFlag

Description:  Enumeration of the different picture info flags.

Comments:

\************************************************************************************************************/
typedef enum tagLH264D_PictureInfoFlag
{
    LH264D_PictureInfoFlag_IDR                        = 0x00000001,
    LH264D_PictureInfoFlag_PICTURENOTREADY            = 0x00000002,
    LH264D_PictureInfoFlag_STREAMINFOFLAGSPRESENT     = 0x00000004,
    LH264D_PictureInfoFlag_CPBDPBDELAYSPRESENT        = 0x00000008,
    LH264D_PictureInfoFlag_PICTURETIMESTAMPRESENT     = 0x00000010,
    LH264D_PictureInfoFlag_RECOVERYPOINTPRESENT       = 0x00000020,
    LH264D_PictureInfoFlag_SEIRBSPPRESENT             = 0x00000040,
} LH264D_PictureInfoFlag;

/************************************************************************************************************\

Enum:         LH264D_StreamInfoUpdateFlag

Description:  Enumeration of the different stream info update flags.

Comments:

\************************************************************************************************************/
typedef enum tagLH264D_StreamInfoUpdateFlag
{
    LH264D_StreamInfoUpdateFlag_GENDATA            = 0x00000001,
    LH264D_StreamInfoUpdateFlag_VUI                = 0x00000002,
    LH264D_StreamInfoUpdateFlag_NALHRD             = 0x00000004,
    LH264D_StreamInfoUpdateFlag_NALBUFFERINGPERIOD = 0x00000008,
    LH264D_StreamInfoUpdateFlag_VCLHRD             = 0x00000010,
    LH264D_StreamInfoUpdateFlag_VCLBUFFERINGPERIOD = 0x00000020
} LH264D_StreamInfoUpdateFlag;

/************************************************************************************************************\

Structure:    LH264D_PictureInfo

Description:  Picture information.

Comments:

\************************************************************************************************************/
typedef struct tagLH264D_PictureInfo
{
    MINT32  eType;                  // force 32 bits cast of LH264D_PictureInfoTypeHeader
    MFLAG32 flPictureStatusFlags;
    MUINT64 hBuffer;                // force 64 bits cast of LBuffer_Handle
    MUINT64 uiAccessUnitId;
    MINT32  eField;                 // force 32 bits cast of LH264_Field
    MUINT64 uiStreamCnt;
    MUINT32 uiStreamId;
    MUINT32 uiCPBConsumedBits;
    MFLAG32 flPictureInfoFlags;
    MFLAG32 flStreamInfoFlags;
    // CPB DPB Delays
    MUINT32 uiCPBRemovalDelay;
    MUINT32 uiDPBOutputDelay;
    // Picture Structure
    MINT32  ePictureStruct;         // force 32 bits cast of LH264_PictureStruct
    MUINT64 auiPictureTimestamp[3];
    // Recovery point
    MUINT16 uiRecoveryFrameCnt;
    MBOOL16 bExactMatchFlag;
    MBOOL16 bBrokenLinkFlag;
    MUINT16 uiChangingSliceGroupIdc;

    // SEI RBSP
    MUINT64 hSEIRbspBuffer;         // force 64 bits cast of LBuffer_Handle
    MUINT32 uiSEIRbspStartOffset;
    MUINT32 uiSEIRbspSize;
    MUINT32 uiSEIRbspContOffset;
    MUINT32 uiSEIRbspContSize;
} LH264D_PictureInfo;


// *********************************************
// 3. Encoder types
// *********************************************
/************************************************************************************************************\

Enum:         LH264E_EncoderMode

Description:  Enumeration of the different encoder modes.

Comments:

\************************************************************************************************************/
typedef enum tagLH264E_EncoderMode
{
    LH264E_EncoderMode_MANUAL                       = 0x00,
    // Rate control modes are structured as (MSB 4 bits => options, LSB 4 BITS => the rate control mode (algorithms))
    // The default rate control algorithm have the LSB 4 bits set to 0.
    // Rate control Options
    LH264E_EncoderMode_RATECONTROL                  = 0x10,
    LH264E_EncoderMode_RATECONTROLLOWLATENCY        = 0x20 | LH264E_EncoderMode_RATECONTROL,
    LH264E_EncoderMode_RATECONTROLDESKTOP           = 0x40 | LH264E_EncoderMode_RATECONTROL,
    // Rate control Mode
    LH264E_EncoderMode_RATECONTROLWINDOWED          = 0x00 | LH264E_EncoderMode_RATECONTROL,
    // Bit Masks
    LH264E_EncoderMode_MASK                         = 0x0F | LH264E_EncoderMode_RATECONTROL,
    LH264E_EncoderOptions_MASK                      = 0xF0 | LH264E_EncoderMode_RATECONTROL,
} LH264E_EncoderMode;

/************************************************************************************************************\

Enum:         LH264E_StartCodeMode

Description:  Enumeration of the different start code modes.

Comments:

\************************************************************************************************************/
typedef enum tagLH264E_StartCodeMode
{
    LH264E_StartCodeMode_NORMAL,
    LH264E_StartCodeMode_ALWAYS4BYTES,
    LH264E_StartCodeMode_NONE
} LH264E_StartCodeMode;

/************************************************************************************************************\

Enum:         LH264E_CreateOptTypeHeader

Description:  Enumeration of the different encode create options types.

Comments:

\************************************************************************************************************/
typedef enum tagLH264E_CreateOptTypeHeader
{
    LH264E_CreateOptTypeHeader_STANDARD
} LH264E_CreateOptTypeHeader;

/************************************************************************************************************\

Enum:         LH264E_CreateOptFlag

Description:  Enumeration of the create options flags.

Comments:

\************************************************************************************************************/
typedef enum tagLH264E_CreateOptFlag
{
    LH264E_CreateOptFlag_ACCESSUNITDELIMITER                = 0x00000001,
    LH264E_CreateOptFlag_GENERATEPICTUREDELAYS              = 0x00000002,
    LH264E_CreateOptFlag_GENERATERECOVERYPOINT              = 0x00000004,
    LH264E_CreateOptFlag_CABAC                              = 0x00000008,
    LH264E_CreateOptFlag_CABACBYTESTUFFING                  = 0x00000010,
    LH264E_CreateOptFlag_LOSSLESSCODINGMODE                 = 0x00000020,
    LH264E_CreateOptFlag_DISABLE8X8TRANSFORM                = 0x00000040,
    LH264E_CreateOptFlag_DISABLEDIRECT8X8INFERENCE          = 0x00000080,
    LH264E_CreateOptFlag_CONSTRAINEDINTRAPREDICTION         = 0x00000100,
    LH264E_CreateOptFlag_COEFFICIENTTHRESHOLDING            = 0x00000200,
    LH264E_CreateOptFlag_VARIABLEFRAMERATE                  = 0x00000400,
    LH264E_CreateOptFlag_DYNAMICPICTURERESOLUTION           = 0x00000800,
    LH264E_CreateOptFlag_IDRONLY                            = 0x00001000,
    LH264E_CreateOptFlag_MBAFF                              = 0x00002000,
    LH264E_CreateOptFlag_BUFFERAUTOSYNC                     = 0x00004000,
    LH264E_CreateOptFlag_PICTUREAUTOSYNCONRELEASE           = 0x00008000,
    LH264E_CreateOptFlag_DECREFPICMARKINGREPETITION         = 0x00010000,
    LH264E_CreateOptFlag_HIGHFRAMERATE                      = 0x00020000,
    LH264E_CreateOptFlag_FORCESINGLESLICE                   = 0x00040000,
} LH264E_CreateOptFlag;

/************************************************************************************************************\

Enum:         LH264E_ConstraintsFlag

Description:  Enumeration of the constraints flags.

Comments:

\************************************************************************************************************/
typedef enum tagLH264E_ConstraintsFlag
{
    LH264E_ConstraintsFlag_SET0    = 0x00000001,
    LH264E_ConstraintsFlag_SET1    = 0x00000002,
    LH264E_ConstraintsFlag_SET2    = 0x00000004,
    LH264E_ConstraintsFlag_SET3    = 0x00000008,
    LH264E_ConstraintsFlag_SET4    = 0x00000010,
    LH264E_ConstraintsFlag_SET5    = 0x00000020,
    LH264E_ConstraintsFlag_PAVC    = 0x00001000,
    LH264E_ConstraintsFlag_SAVC    = 0x00002000,
} LH264E_ConstraintsFlag;

/************************************************************************************************************\

Structure:    LH264E_CreateOpt

Description:  Options for encode context creation.

Comments:

\************************************************************************************************************/
typedef struct tagLH264E_CreateOpt
{
    MINT32  eType;              // force 32 bits cast of LH264E_CreateOptTypeHeader
    MUINT64 hDTInput;           // force 64 bits cast of LDeviceThread_Handle
    MUINT64 hDTOutput;          // force 64 bits cast of LDeviceThread_Handle
    MFLAG32 flCreateOptFlags;
    MINT32  eEncoderMode;       // force 32 bits cast of LH264E_EncoderMode
    MINT32  ePriority;          // force 32 bits cast of LH264_Priority
    MINT32  eProfile;           // force 32 bits cast of LH264_Profile
    MINT32  eLevel;             // force 32 bits cast of LH264_Level
    MFLAG32 flConstraints;
    MUINT32 uiGOPOffset;
    MINT32  eStartCodeMode;     // force 32 bits cast of LH264E_StartCodeMode
    MUINT32 uiOutputBufferAUExtraSize; 
    MUINT32 uiAutoSyncTimeOut;  // Used only with option LH264E_CreateOptFlag_BUFFERAUTOSYNC
} LH264E_CreateOpt;

/************************************************************************************************************\

Enum:         LH264E_EncodeOptTypeHeader

Description:  Enumeration of the different encode options types.

Comments:

\************************************************************************************************************/
typedef enum tagLH264E_EncodeOptTypeHeader
{
    LH264E_EncodeOptTypeHeader_STANDARD
} LH264E_EncodeOptTypeHeader;

/************************************************************************************************************\

Enum:         LH264E_PictureFlag

Description:  Encode options.

Comments:

\************************************************************************************************************/
typedef enum tagLH264E_PictureFlag
{
    LH264E_PictureFlag_PICTURETIMINGPRESENT                = 0x00000001,
    LH264E_PictureFlag_RECOVERYPOINTPRESENT                = 0x00000002,
    LH264E_PictureFlag_SEIRBSPPRESENT                      = 0x00000004,
    LH264E_PictureFlag_ENDOFSEQUENCE                       = 0x00000008,
    LH264E_PictureFlag_SEIDIRECTCOPY                       = 0x00000010
} LH264E_PictureFlag;

/************************************************************************************************************\

Enum:         LH264E_DFDisableIdc

Description:  Deblocking filter disable IDC

Comments:

\************************************************************************************************************/
typedef enum tagLH264E_DFDisableIdc
{
    LH264E_DFDisableIdc_ENABLED      = 0,
    LH264E_DFDisableIdc_DISABLED     = 1,
    LH264E_DFDisableIdc_SLICEENABLED = 2
} LH264E_DFDisableIdc;

/************************************************************************************************************\

Structure:    LH264E_EncodeOpt

Description:  Options for encode picture.

Comments:

\************************************************************************************************************/
typedef struct tagLH264E_EncodeOpt
{
    MINT32   eType;                 // force 32 bits cast of LH264E_EncodeOptTypeHeader
    MUINT64  hPicture;              // force 64 bits cast of LBuffer_Handle
    MUINT64  uiPictureId;
    MINT32   ePictureType;          // force 32 bits cast of LH264_PictureType
    MFLOAT32 fQP;
    MUINT32  uiSliceIndex;
    MFLAG32  flPictureFlags;
    // Picture timing
    MINT32   ePictureStruct;        // force 32 bits cast of LH264_PictureStruct
    MUINT64  auiPictureTimestamp[3];
    // Recovery point
    MUINT32  uiRecoveryFrameCnt;
    MBOOL32  bExactMatchFlag;
    MBOOL32  bBrokenLinkFlag;
    // SEI RBSP
    MUINT64  hSEIRbspBuffer;        // force 64 bits cast of LBuffer_Handle
    MUINT32  uiSEIRbspStartOffset;
    MUINT32  uiSEIRbspSize;
} LH264E_EncodeOpt;

/************************************************************************************************************\

Enum:         LH264E_NaluInfoTypeHeader

Description:  Enumeration of the different Nalu info types.

Comments:

\************************************************************************************************************/
typedef enum tagLH264E_NaluInfoTypeHeader
{
    LH264E_NaluInfoTypeHeader_STANDARD
} LH264E_NaluInfoTypeHeader;

/************************************************************************************************************\

Enum:         LH264E_NaluStatusFlag

Description:  Enumeration of the Nalu status.

Comments:

\************************************************************************************************************/
typedef enum tagLH264E_NaluStatusFlag
{
    LH264E_NaluStatusFlag_FATALFIRMWAREERROR            = 0x00000001,
    LH264E_NaluStatusFlag_FATALSTREAMERROR              = 0x00000002,
    LH264E_NaluStatusFlag_MISSINGPARAMETERS             = 0x00000004,
    LH264E_NaluStatusFlag_UNSUPPORTEDPARAMETERS         = 0x00000008,
    LH264E_NaluStatusFlag_PICTURESSKIPPED               = 0x00000010,
    LH264E_NaluStatusFlag_PICTUREREPROCESSED            = 0x00000020,
    LH264E_NaluStatusFlag_PSKIP                         = 0x00000040,
    LH264E_NaluStatusFlag_BIGPICTURE                    = 0x00000080,
    LH264E_NaluStatusFlag_CPBOVERFLOW                   = 0x00000100,
    LH264E_NaluStatusFlag_CPBUNDERFLOW                  = 0x00000200,
    LH264E_NaluStatusFlag_SCENECHANGE                   = 0x00000400,
    LH264E_NaluStatusFlag_STREAMERROR                   = 0x00000800,
    LH264E_NaluStatusFlag_AUFENCE                       = 0x00001000,
    LH264E_NaluStatusFlag_QPMINSAT                      = 0x00002000,
    LH264E_NaluStatusFlag_QPMAXSAT                      = 0x00004000
} LH264E_NaluStatusFlag;

/************************************************************************************************************\

Structure:    LH264E_NaluInfo

Description:  Nalu information.

Comments:

\************************************************************************************************************/
typedef struct tagLH264E_NaluInfo
{
    MINT32   eType;              // force 32 bits cast of LH264E_NaluInfoTypeHeader
    MFLAG32  flNaluStatusFlags;
    MUINT64  hBuffer;            // force 64 bits cast of LBuffer_Handle
    MUINT32  uiStartOffset;
    MUINT32  uiSize;
    MUINT64  uiNaluCnt;
    MUINT64  uiAUCnt;
    MUINT64  uiPictureId;
    MINT32   ePictureType;       // force 32 bits cast of LH264_PictureType
    MINT32   eNaluType;          // force 32 bits cast of LH264_NaluType
    MUINT32  uiHierarchyLevel;
    MUINT64  hPictureReleased;   // force 64 bits cast of LBuffer_Handle
    MFLOAT32 fSumSquaredError;
} LH264E_NaluInfo;

// -----------------------------------------------------------------------------------------------------------
//                                 F U N C T I O N   D E C L A R A T I O N S
// -----------------------------------------------------------------------------------------------------------

#if defined (__cplusplus)
extern "C" {
#endif

// ******************************************
// Decoder Functions.
// ******************************************

/************************************************************************************************************\

Function:       LH264D_Create

Description:    Create a decoder video context.

Parameters:     IN  hDev              Handle to a device object.
                IN  peDecodeCreateOpt Options for creating a decode context (passed by type header).
                OUT phDecode          Decode context handle returned.

Return Value:   LStatus_OK                                  Function completed successfully.
                LStatus_OK_STREAM_NOT_LOWLATENCY            Function completed, but not in Low Latency.
                LStatus_FAIL                                Function failed.
                LStatus_INVALID_PARAM                       Function failed. Invalid parameters.
                LStatus_OUT_OF_RESOURCES                    Function failed. Resources not available.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LH264D_Create
(
    LDevice_Handle              hDev,
    LH264D_CreateOptTypeHeader* peDecodeCreateOpt,
    LH264D_Handle*              phDecode
);

/************************************************************************************************************\

Function:       LH264D_Destroy

Description:    Destroy a decoder video context.

Parameters:     IN hDecode  Handle to a decode context.

Return Value:   LStatus_OK                                  Function completed successfully.
                LStatus_FAIL                                Function failed.
                LStatus_INVALID_PARAM                       Function failed. Invalid parameters.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LH264D_Destroy
(
    LH264D_Handle hDecode
);

/************************************************************************************************************\

Function:       LH264D_DecodeNALBlob

Description:    Sends encoded data to the decoder.

Parameters:     IN  hDecode             Handle to a decode context.
                IN  peDecodeOpt         Options for decoding the NAL blob (passed by type header).
                OUT puiBytesConsumed    Number of bytes consumed in the NAL blob

Return Value:   LStatus_OK                                  Function completed successfully.
                LStatus_OK_STREAM_NOT_LOWLATENCY            Function completed, but not in Low Latency.
                LStatus_INCOMPLETE_COMMAND                  Function has partially completed the command (see puiBytesConsumed).
                LStatus_FAIL                                Function failed.
                LStatus_DEVICETHREAD_COMMAND_QUEUE_FULL     Function failed. Device thread is full.
                LStatus_INVALID_PARAM                       Function failed. Invalid parameters.
                LStatus_OUT_OF_RESOURCES                    Function failed. Resources not available.
                LStatus_TIMEOUT                             Function failed. A buffer with autosynchronization is busy.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LH264D_DecodeNALBlob
(
    LH264D_Handle               hDecode,
    LH264D_DecodeOptTypeHeader* peDecodeOpt,
    MUINT32*                    puiBytesConsumed
);

/************************************************************************************************************\

Function:       LH264D_FlushDTInput

Description:    Flush the decoder input device-threads.

Parameters:     IN hDecode      Handle to a decode context.

Return Value:   LStatus_OK                                  Function completed successfully.
                LStatus_FAIL                                Function failed.
                LStatus_DEVICETHREAD_COMMAND_QUEUE_FULL     Function failed. Device thread is full.
                LStatus_INVALID_PARAM                       Function failed. Invalid parameters.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LH264D_FlushDTInput
(
    LH264D_Handle hDecode
);

/************************************************************************************************************\

Function:       LH264D_ResetSequence

Description:    Send a Reset-Sequence command to the decoder to terminate current sequence and jump further
                in the stream. This command pauses the decoding pipeline until all the previous decoding 
                commands are executed, it also forces all the picture to be output and then resets the DPB 
                to prepares the codec to jump further into the encoded byte stream.

Parameters:     IN hDecode      Handle to a decode context.

Return Value:   LStatus_OK                                  Function completed successfully.
                LStatus_FAIL                                Function failed.
                LStatus_DEVICETHREAD_COMMAND_QUEUE_FULL     Function failed. Device thread is full.
                LStatus_INVALID_PARAM                       Function failed. Invalid parameters.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LH264D_ResetSequence
(
    LH264D_Handle hDecode
);

/************************************************************************************************************\

Function:       LH264D_EndStream

Description:    Ends the stream before destroying the context.

Parameters:     IN hDecode      Handle to a decode context.

Return Value:   LStatus_OK                                  Function completed successfully.
                LStatus_FAIL                                Function failed.
                LStatus_DEVICETHREAD_COMMAND_QUEUE_FULL     Function failed. Device thread is full.
                LStatus_INVALID_PARAM                       Function failed. Invalid parameters.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LH264D_EndStream
(
    LH264D_Handle hDecode
);

/************************************************************************************************************\

Function:       LH264D_GetNextPicture

Description:    Sends encoded data to the decoder.

Parameters:     IN hDecode          Handle to a decode context.
                IN pePictureInfo    Returns information on the current picture (passed by type header).
                IN uiTimeOut        Delay, in milliseconds, for a new picture:
                                          - 0                = no wait,
                                          - LIFINITE_TIMEOUT = wait until a picture is available

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_SEQUENCE_RESET  The sequence has been reset.
                LStatus_END_OF_STREAM   No additional pictures available for the current stream.
                LStatus_TIMEOUT         Timeout expired. No new data available.
                LStatus_FAIL            Function failed.
                LStatus_INVALID_PARAM   Function failed. Invalid parameters.
                LStatus_STREAM_NOT_INITIALIZED Stream initialization not yet completed/done
                LSTATUS_CRITICAL_END_OF_STREAM stream Critical failure generating an end of stream

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LH264D_GetNextPicture
(
    LH264D_Handle                 hDecode,
    LH264D_PictureInfoTypeHeader* pePictureInfo,
    MUINT32                       uiTimeOut
);

/************************************************************************************************************\

Function:       LH264D_GetStreamInfo

Description:    Retrieves the information on the current stream.

Parameters:     IN hDecode          Handle to a decode context.
                IN peStreamInfo     Information update on the current stream (passed by type header).

Return Value:   LStatus_OK                  Function completed successfully.
                LStatus_INFO_NOT_AVAILABLE  No information currently available for this stream.
                LStatus_FAIL                Function failed.
                LStatus_INVALID_PARAM       Function failed. Invalid parameters.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LH264D_GetStreamInfo
(
    LH264D_Handle               hDecode,
    LH264_StreamInfoTypeHeader* peStreamInfo
);

/************************************************************************************************************\

Function:       LH264D_ReleasePicture

Description:    Releases a decoded picture.

Parameters:     IN hDecode          Handle to a decode context.
                IN pePictureInfo    Picture to be released (passed by type header).

Return Value:   LStatus_OK                                  Function completed successfully.
                LStatus_FAIL                                Function failed.
                LStatus_INVALID_PARAM                       Function failed. Invalid parameters.
                LStatus_DEVICETHREAD_COMMAND_QUEUE_FULL     Function failed. Device thread is full.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LH264D_ReleasePicture
(
    LH264D_Handle                 hDecode,
    LH264D_PictureInfoTypeHeader* pePictureInfo
);

// ******************************************
// Encoder Functions.
// ******************************************

/************************************************************************************************************\

Function:       LH264E_Create

Description:    Create an encoder video context.

Parameters:     IN  hDev               Handle to a device object.
                IN  peEncodeCreateOpt  Options for creating an encode context (passed by type header).
                OUT phEncode           Encode context handle returned.

Return Value:   LStatus_OK                                  Function completed successfully.
                LStatus_FAIL                                Function failed.
                LStatus_INVALID_PARAM                       Function failed. Invalid parameters.
                LStatus_UNSUPPORTED                         Function failed. The function is not supported.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LH264E_Create
(
    LDevice_Handle              hDev,
    LH264E_CreateOptTypeHeader* peEncodeCreateOpt,
    LH264E_Handle*              phEncode
);

/************************************************************************************************************\

Function:       LH264E_Destroy

Description:    Destroy an encoder video context.

Parameters:     IN hEncode  Handle to an encode context.

Return Value:   LStatus_OK                                  Function completed successfully.
                LStatus_FAIL                                Function failed.
                LStatus_INVALID_PARAM                       Function failed. Invalid parameters.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LH264E_Destroy
(
    LH264E_Handle hEncode
);

/************************************************************************************************************\

Function:       LH264E_GetNbSliceRange

Description:    Returns the minimum and maximum supported number of slices.

Parameters:     IN  hEncode          Handle to an encode context.
                OUT puiMinNbSlices   Minimum number of slices supported.
                OUT puiMaxNbSlices   Maximum number of slices supported.

Return Value:   LStatus_OK                                  Function completed successfully.
                LStatus_FAIL                                Function failed.
                LStatus_INVALID_PARAM                       Function failed. Invalid parameters.

Comments:       This function should be called after calling UpdateStreamInfo() with a SEQUENCE DATA and 
                SUBSEQUENCE DATA stream information.

\************************************************************************************************************/
LAPI LStatus LH264E_GetNbSliceRange
(
    LH264E_Handle hEncode,
    MUINT32*      puiMinNbSlices,
    MUINT32*      puiMaxNbSlices
);

/************************************************************************************************************\

Function:       LH264E_UpdateStreamInfo

Description:    Update the video stream settings.

Parameters:     IN hEncode          Handle to an encode context.
                IN peStreamInfo     New options and settings to apply to the video stream (passed by type header).

Return Value:   LStatus_OK                                  Function completed successfully.
                LStatus_FAIL                                Function failed.
                LStatus_DEVICETHREAD_COMMAND_QUEUE_FULL     Function failed. Device thread is full.
                LStatus_INVALID_PARAM                       Function failed. Invalid parameters.
                LStatus_REORDERED_SEQUENCE_PENDING          Function can't apply the new information.
                                                            Information was sent in the middle of a reordered sequence.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LH264E_UpdateStreamInfo
(
    LH264E_Handle               hEncode,
    LH264_StreamInfoTypeHeader* peStreamInfo
);

/************************************************************************************************************\

Function:       LH264E_PrepareToEncode

Description:    Finalize the encoder initialization just before starting encoding pictures.

Parameters:     IN  hEncode         Handle to an encode context.

Return Value:   LStatus_OK                                  Function completed successfully.
                LStatus_FAIL                                Function failed.
                LStatus_DEVICETHREAD_COMMAND_QUEUE_FULL     Function failed. Device thread is full.
                LStatus_INVALID_PARAM                       Function failed. Invalid parameters.
                LStatus_OUT_OF_RESOURCES                    Function failed. Resources not available.

Comments:       This call must immediately precede the first call to LH264E_Encode().

\************************************************************************************************************/
LAPI LStatus LH264E_PrepareToEncode
(
    LH264E_Handle               hEncode
);

/************************************************************************************************************\

Function:       LH264E_Encode

Description:    Sends a picture to be encoded.

Parameters:     IN  hEncode         Handle to an encode context.
                IN  peEncodeOpt     Options to encode the picture (passed by type header).

Return Value:   LStatus_OK                                  Function completed successfully.
                LStatus_FAIL                                Function failed.
                LStatus_DEVICETHREAD_COMMAND_QUEUE_FULL     Function failed. Device thread is full.
                LStatus_INVALID_PARAM                       Function failed. Invalid parameters.
                LStatus_REORDERED_SEQUENCE_PENDING          Function can't apply the new information.
                                                            Information was sent in the middle of a reordered sequence.
                LStatus_OUT_OF_RESOURCES                    Function failed. Resources not available.
                LStatus_TIMEOUT                             Function failed. A buffer with autosynchronization is busy.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LH264E_Encode
(
    LH264E_Handle               hEncode,
    LH264E_EncodeOptTypeHeader* peEncodeOpt
);

/************************************************************************************************************\

Function:       LH264E_FlushDTInput

Description:    Flush the encoder input device-threads.

Parameters:     IN hEncode      Handle to an encode context.

Return Value:   LStatus_OK                                  Function completed successfully.
                LStatus_FAIL                                Function failed.
                LStatus_DEVICETHREAD_COMMAND_QUEUE_FULL     Function failed. Device thread is full.
                LStatus_INVALID_PARAM                       Function failed. Invalid parameters.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LH264E_FlushDTInput
(
    LH264E_Handle hEncode
);

/************************************************************************************************************\

Function:       LH264E_EndStream

Description:    Ends the stream before destroying the context.

Parameters:     IN hEncode         Handle to an encode context.

Return Value:   LStatus_OK                                  Function completed successfully.
                LStatus_FAIL                                Function failed.
                LStatus_INVALID_PARAM                       Function failed. Invalid parameters.
                LStatus_DEVICETHREAD_COMMAND_QUEUE_FULL     Function failed. Device thread is full.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LH264E_EndStream
(
    LH264E_Handle hEncode
);

/************************************************************************************************************\

Function:       LH264E_GetNextNalu

Description:    Retrieves the following NALU.

Parameters:     IN  hEncode         Handle to an encode context.
                OUT peNaluInfo      Returns information associated with an encoded NALU (passed by type).
                IN  uiTimeOut       Delay, in milliseconds, for a new picture:
                                                - 0                 = no wait
                                                - LINFINITE_TIMEOUT = wait until a picture is available

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_END_OF_STREAM   No additional NALUs available for the current stream.
                LStatus_TIMEOUT         Timeout expired. No new data is available yet.
                LStatus_FAIL            Function failed.
                LStatus_INVALID_PARAM   Function failed. Invalid parameters.
                LStatus_STREAM_NOT_INITIALIZED Stream initialization not yet completed/done
                LSTATUS_CRITICAL_END_OF_STREAM stream Critical failure generating an end of stream

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LH264E_GetNextNalu
(
    LH264E_Handle                     hEncode,
    LH264E_NaluInfoTypeHeader*        peNaluInfo,
    MUINT32                           uiTimeOut
);

/************************************************************************************************************\

Function:       LH264E_ReleaseNalu

Description:    Releases a NALU.

Parameters:     IN hEncode         Handle to an encode context.
                IN peNaluInfo      NALU to be released (passed by type).

Return Value:   LStatus_OK                                  Function completed successfully.
                LStatus_FAIL                                Function failed.
                LStatus_INVALID_PARAM                       Function failed. Invalid parameters.
                LStatus_DEVICETHREAD_COMMAND_QUEUE_FULL     Function failed. Device thread is full.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LH264E_ReleaseNalu
(
    LH264E_Handle              hEncode,
    LH264E_NaluInfoTypeHeader* peNaluInfo
);

// ******************************************
// Global Functions.
// ******************************************

/************************************************************************************************************\

Function:       LH264_GetGlobalInfo

Description:    Fetches global statistics and information in the VC architecture.

Parameters:     IN hDev             Handle to an encode or a decode context.
                IN peGlobalInfo     Global statistics and information on the codec (passed by type header).

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_FAIL            Function failed.
                LStatus_INVALID_PARAM   Function failed. Invalid parameters.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LH264_GetGlobalInfo
(
    LDevice_Handle              hDev,
    LH264_GlobalInfoTypeHeader* peGlobalInfo
);

/************************************************************************************************************\

Function:       LH264_CreateFirmwareTraceOptions

Description:    Creates a trace option

Parameters:     hDev                Handle of the device to attach the trace options to.
                phTraceOptions      The new trace options, if sucessfull

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_FAIL            Function failed.
                LStatus_INVALID_PARAM   Function failed. Invalid parameters.

Comments:       

\************************************************************************************************************/
LAPI LStatus LH264_CreateFirmwareTraceOptions(
    LDevice_Handle              hDev,
    LH264_TraceOptions*         phTraceOptions);

/************************************************************************************************************\

Function:       LH264_ApplyFirmwareTraceOptions

Description:    Applies the trace options

Parameters:     hTraceOptions       The trace options to use

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_FAIL            Function failed.
                LStatus_INVALID_PARAM   Function failed. Invalid parameters.

Comments:       

\************************************************************************************************************/
LAPI LStatus LH264_ApplyFirmwareTraceOptions(
    LH264_TraceOptions          hTraceOptions);

/************************************************************************************************************\

Function:       LH264_DestroyFirmwareTraceOptions

Description:    Destroys a trace options

Parameters:     hTraceOptions       The trace options to use

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_FAIL            Function failed.
                LStatus_INVALID_PARAM   Function failed. Invalid parameters.

Comments:       

\************************************************************************************************************/
LAPI LStatus LH264_DestroyFirmwareTraceOptions(
    LH264_TraceOptions          hTraceOptions);

/************************************************************************************************************\

Function:       LH264_SetFirmwareGlobalTraceLevel

Description:    Set the global trace level.

Parameters:     hTraceOptions       The trace options to use
                uiTraceLevel        The trace level to apply.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_FAIL            Function failed.
                LStatus_INVALID_PARAM   Function failed. Invalid parameters.

Comments:       This effectively wipe out the previous module trace levels and replace it by a single
                level.

\************************************************************************************************************/
LAPI LStatus LH264_SetFirmwareGlobalTraceLevel(
    LH264_TraceOptions          hTraceOptions,
    MUINT32                     uiTraceLevel);

/************************************************************************************************************\

Function:       LH264_SetFirmwareModuleTraceLevel

Description:    Set the trace level of a VC firmware module.

Parameters:     hTraceOptions       The trace options to use
                szFWModuleName      The module name to apply the level to.
                uiTraceLevel        The trace level to apply.

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_FAIL            Function failed.
                LStatus_INVALID_PARAM   Function failed. Invalid parameters.

Comments:       

\************************************************************************************************************/
LAPI LStatus LH264_SetFirmwareModuleTraceLevel(
    LH264_TraceOptions          hTraceOptions,
    const char*                 szFWModuleName,
    MUINT32                     uiTraceLevel);


/************************************************************************************************************\

Function:       LH264D_GetDTInputLastSubmissionTag

Description:    Returns the last submitted tag for intput device thread

Parameters:     IN      hDecode             Handle to an decode context.
                OUT     puiSubmissionTag    allocation where the value will be written
                OUT     puiDTInput          DeviceThread used to submit the last command

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed. Invalid parameters..

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LH264D_GetDTInputLastSubmissionTag(LH264D_Handle          hDecode,
                                                MUINT64*               puiLastSubmissionTag,
                                                LDeviceThread_Handle*  phDTInput);

/************************************************************************************************************\

Function:       LH264D_GetDTOutputLastSubmissionTag

Description:    Returns the last submitted tag for output device thread

Parameters:     IN      hDecode             Handle to an decode context.
                OUT     puiSubmissionTag    allocation where the value will be written
                OUT     phDTOutput          DeviceThread used to submit the last command

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed. Invalid parameters..

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LH264D_GetDTOutputLastSubmissionTag(LH264D_Handle         hDecode,
                                                 MUINT64*              puiLastSubmissionTag,
                                                 LDeviceThread_Handle* phDTOutput);

/************************************************************************************************************\

Function:       LH264E_GetDTInputLastSubmissionTag

Description:    Returns the last submitted tag for input device thread

Parameters:     IN      hEncode             Handle to an encode context.
                OUT     puiSubmissionTag    allocation where the value will be written
                OUT     phDTInput           DeviceThread used to submit the last command

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed. Invalid parameters..

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LH264E_GetDTInputLastSubmissionTag(LH264E_Handle          hEncode,
                                                MUINT64*               puiSubmissionTag,
                                                LDeviceThread_Handle*  phDTInput);

/************************************************************************************************************\

Function:       LH264E_GetDTOutputLastSubmissionTag

Description:    Returns the last submitted tag for output device thread

Parameters:     IN      hEncode             Handle to an encode context.
                OUT     puiSubmissionTag    allocation where the value will be written
                OUT     phDTOutput          DeviceThread used to submit the last command

Return Value:   LStatus_OK              Function completed successfully.
                LStatus_INVALID_PARAM   Function failed. Invalid parameters..


Comments:       None.

\************************************************************************************************************/
LAPI LStatus LH264E_GetDTOutputLastSubmissionTag(LH264E_Handle          hEncode,
                                                 MUINT64*               puiSubmissionTag,
                                                 LDeviceThread_Handle*  phDTOutput);

// -----------------------------------------------------------------------------------------------------------
//                       I N L I N E S   A N D   T E M P L A T E   D E F I N I T I O N S
// -----------------------------------------------------------------------------------------------------------

// Generic Helper macros (Decoder & Encoder).
#define Declare_LH264_GlobalInfo                        ( oName ) \
                LH264_GlobalInfo                          oName = { LH264_GlobalInfoTypeHeader_STANDARD };
#define Declare_LH264_StreamInfoDecoderGenData          ( oName ) \
                LH264_StreamInfoDecoderGenData            oName = { LH264_StreamInfoTypeHeader_DECODERGENDATA };
#define Declare_LH264_StreamInfoEncoderSequenceData     ( oName ) \
                LH264_StreamInfoEncoderSequenceData       oName = { LH264_StreamInfoTypeHeader_ENCODERSEQUENCEDATA };
#define Declare_LH264_StreamInfoEncoderNbSlices         ( oName ) \
                LH264_StreamInfoEncoderNbSlices           oName = { LH264_StreamInfoTypeHeader_ENCODERNBSLICES };
#define Declare_LH264_StreamInfoEncoderSubSequenceData  ( oName ) \
                LH264_StreamInfoEncoderSubSequenceData    oName = { LH264_StreamInfoTypeHeader_ENCODERSUBSEQUENCEDATA };
#define Declare_LH264_StreamInfoVUI                     ( oName ) \
                LH264_StreamInfoVUI                       oName = { LH264_StreamInfoTypeHeader_VUI };
#define Declare_LH264_StreamInfoHRD                     ( oName ) \
                LH264_StreamInfoHRD                       oName = { LH264_StreamInfoTypeHeader_HRD };
#define Declare_LH264_StreamInfoBufferingPeriod         ( oName ) \
                LH264_StreamInfoBufferingPeriod           oName = { LH264_StreamInfoTypeHeader_BUFFERINGPERIOD };
#define Declare_LH264_StreamInfoScalingList             ( oName ) \
                LH264_StreamInfoScalingList               oName = { LH264_StreamInfoTypeHeader_SPSSCALINGLIST };
// Decoder Helper macros.
#define Declare_LH264D_CreateOpt                        ( oName ) \
                LH264D_CreateOpt                          oName = { LH264D_CreateOptTypeHeader_STANDARD };
#define Declare_LH264D_DecodeOpt                        ( oName ) \
                LH264D_DecodeOpt                          oName = { LH264D_DecodeOptTypeHeader_STANDARD };
#define Declare_LH264D_PictureInfo                      ( oName ) \
                LH264D_PictureInfo                        oName = { LH264D_PictureInfoTypeHeader_STANDARD };
// Encoder Helper macros.
#define Declare_LH264E_CreateOpt                        ( oName ) \
                LH264E_CreateOpt                          oName = { LH264E_CreateOptTypeHeader_STANDARD };
#define Declare_LH264E_EncodeOpt                        ( oName ) \
                LH264E_EncodeOpt                          oName = { LH264E_EncodeOptTypeHeader_STANDARD };
#define Declare_LH264E_NaluInfo                         ( oName ) \
                LH264E_NaluInfo                           oName = { LH264E_NaluInfoTypeHeader_STANDARD };

#if defined (__cplusplus)
}
#endif

#pragma pack(pop)

#endif  // #ifndef INC_LH264VIDEOCODEC_H
